function QDB3(tifPathName, tifFileName, fitPathName, fitFileName)
% function QDB3
%    or 
% function QDB3(fitPathName, fitFileName, tifPathName, tifFileName)
%
% Matlab code to perform QDB3 analysis on tif movies. Please change the
% parameters in the code if necessary.
%
%
% Note that this code makes uses of the output (*.spotFits.txt) from 
% gSHRImP (Simonson, Paul D., Eli Rothenberg, and Paul R. Selvin.
% Single-Molecule-Based Super-Resolution Images in the Presence of
% Multiple Fluorophores. Nano Letters 11, no. 11 (2011): 5090-5096.
% doi:10.1021/nl203560r.) The program is available from Dr. Paul R. Selvin.
% 
% Author: Yong Wang @ Selvin Lab @ UIUC
% Publication: (Wang, Yong, Gilbert Fruhwirth, En Cai, Tony Ng, and Paul R.
% Selvin. 3D Super-Resolution Imaging with Blinking Quantum Dots. Nano
% Letters 13, no. 11 (November 13, 2013): 5233-5241.
% doi:10.1021/nl4026665.)
%
% (Still working on C/C++ version for faster processing, and
% ImageJ plugins for a user-friendly version).


global spots

% refractive index mismatch for real experiments, need to change according
% to experimental conditions (i.e., buffer, objective, etc)
factorRIMismatch=0.79;
% when testing for simulated data, no mismatch at all, use 1.0
% factorRIMismatch=1.0;

if nargin~=4
    % get tif file name
    [tifFileName,tifPathName,filterIndex] = uigetfile('*.tif','Select TIF file');      
    % get spotFits.txt file
    [fitFileName,fitPathName,filterIndex] = uigetfile('*spotFits.txt','Select spotFits file');    
end
fullTifFileName=[tifPathName,tifFileName];  
fullFitFileName=[fitPathName,fitFileName];

% get the average image for drawing (presentation purpose) at the end
fprintf('Loading the AVE image ...\n');
aveRawImgFileName=[fullTifFileName,'.stats.ave'];
if exist(aveRawImgFileName,'file')~=2
    aveImg=imgStats(fullTifFileName,'ave');
else
    aveImg=load(aveRawImgFileName);
end

% load fitting parameters
spots=load(fullFitFileName); % load fitted spots
spots=selectGoodSpots(spots); % apply additional criteria
n=size(spots,1); % get number of spots

x=spots(:,2);
y=spots(:,3);
sx=spots(:,4);
sy=spots(:,5);


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% choose either fitted curve or use the curve to generate calibration data
% beforehand (calibration.dat)

% using functions to get ellipiticity, need to provide the function for
% the fitted curve: ell(z)
% zrange=-400.0:0.01:400.00;
% ellipticity=ell(zrange); 

% load ellipiticity directly, need to provide 'calibration.dat'
foo=load('calibration.dat');
zrange=foo(:,1);
ellipticity=foo(:,2);

% calc z from calibration curve
r=sy./sx;
z=zeros(n,1);
for i=1:n    
    e=1.0*floor(r(i)*100)/100.0;    
    z(i)=mean(zrange(abs(ellipticity-e)<0.01))/100.00*factorRIMismatch;
end

% save results
frame=(1:n)';
data=[frame,x,y,z];
xyzfile=[fullFitFileName,'.QDB3'];
save(xyzfile,'data','-ascii');

% plot the results
figure(1); clf;
sz=size(aveImg);
imagesc(aveImg,[min(aveImg(:)) max(aveImg(:))/2]);
colormap(gray);
hold on;
cx=x+1;
cy=sz(1)-y;
cz=z;

Dz=max(cz(:))-min(cz(:));
zshift=min(cz(:));
for i=1:max(size(cx))
    if ~isnan(cz(i))
        hold on;
        c=1.0*(cz(i)-min(cz(:)))/Dz;
        crgb=[c,1-c,0];
        plot3(cx(i), cy(i), cz(i)-zshift,'o', 'MarkerFaceColor', crgb, 'MarkerEdgeColor', crgb);   
    end
end
axis([1 sz(2) 1 sz(1)]);
view(0,90);
% view(3);
axis equal;
axis([1 sz(2) 1 sz(1)]);
xlabel('X');
ylabel('Y');
zlabel('Z');


end

function goodSpots=selectGoodSpots(spots)
IndexHeight=1;
IndexX=2;
IndexY=3;
IndexSX=4;
IndexSY=5;
IndexBG=6;
IndexTiltAngle=7;
IndexHeightError=8;
IndexXError=9;
IndexYError=10;
IndexSXError=11;
IndexSYError=12;


foo=spots(:, IndexHeight);
foo=sort(foo,'ascend');

% Criteria: peak height (i.e. intensity)
% change the values according to the intensity of a single qdot under the
% experimental conditions. In practice, it is better to monitor the
% intensity of a single qdot for a long time (under the same conditions)
% such that one has an idea how the intensity should look like.
minHeight=4500;
maxHeight=5500;
maxHeightError=maxHeight*0.1;

% Criteria: tilt of the PSF
% The tilting should be along either x-axis or y-axis.
% If the PSF is not stretched along x/y, there is something wrong with the
% experiment. Adjust cylindrical lens or camera or setup alignment.
% Set maxDifTiltAngle to a small value to tolerate some fitting errors.
centerTiltAngle=0.0; % this value should be fixed (re-align the setup, adjust experimental conditions)
maxDifTiltAngle=1.0; % this value should be always small (keep < 5)
% project the tilting angle to [0-90]
spots(:,IndexTiltAngle)=mod(mod(spots(:,IndexTiltAngle),360),90);

% Criteria: ellipticity
minEllipticity = 0.7; % change this value according to the calibration
maxEllipticity = 1.4; % change this value according to the calibration

% calculate the ellipticity from fittings
sx=spots(:,4);
sy=spots(:,5);
r=sy./sx;

goodSpots=spots(spots(:,IndexHeight)>=minHeight & ...
    spots(:,IndexHeight)<=maxHeight &...
    spots(:,IndexHeightError)<=maxHeightError &...
    abs(spots(:,IndexTiltAngle)-centerTiltAngle)<=maxDifTiltAngle &...
    r(:)>=minEllipticity &... 
    r(:)<=maxEllipticity ...
    , :);
end


function stats=imgStats(file, mode, numframe)
% function stats=imgStats(file, mode, numframe) get the statistics of a
% series of images in 'file'
% Inputs:
%   - file: image file name (string), e.g. 'testfile.tif'
%   - mode: (optional) which to calculate (string), e.g. 'max', 'ave',
%   'min', 'all'  (default: 'max')
%   - numframe: (optional) number of frames used for calculate the stats,
%   e.g. 500 (default: total number of frames from file)
%
% Revision History:
%   - Created by Yong Wang @ April 2012
%   - Modified by Yong Wang @ June 2012: add different modes

if nargin<3, numframe=-1; end;
if nargin<2, mode='max'; end;
if nargin<1, error('Tell me the tif file name, please!'); end;

disp(['Trying to get information from ', file, '...']);
A=imfinfo(file);
width=A(1).Width;
height=A(1).Height;

if (numframe<0) numframe=numel(A); end;

aveimg=zeros(height,width);
maximg=zeros(height,width);
minimg=zeros(height,width);

for stack=1:numframe
    cimage=double(imread(file,stack));
    maximg=max(maximg,cimage);
    minimg=min(minimg,cimage);
    aveimg=aveimg+cimage;

    if(mod(stack, 100)==1) 
        disp(['Reading stack ', num2str(stack), ' / ', num2str(numframe)]);
    end
end
aveimg=aveimg/numframe;

disp('Reading is done! Saving data ...');

allstats=struct();
allstats.maximg=maximg;
allstats.minimg=minimg;
allstats.aveimg=aveimg;

if mode=='max'
    stats=allstats.maximg;
elseif mode=='min'
    stats=allstats.minimg;
elseif mode=='ave'
    stats=allstats.aveimg; 
else
    mode='all'
    stats=allstats; 
end;

save([file, '.allstats'], 'allstats');
save([file, '.stats.',mode], 'stats','-ascii');

end

